from enum import Enum
from datetime import datetime
from pydantic import BaseModel, Field
from typing import Optional

# ====================================================
# Enums to capture timesheet types and statuses
# ====================================================
class TimesheetType(str, Enum):
    scheduled = "scheduled"   # Pre-planned shifts
    on_demand = "on_demand"     # Ad-hoc or immediate assignments

class TimesheetStatus(str, Enum):   
    pending = "pending"       # Timesheet entry created but not yet started
    active = "active"         # Currently in progress
    completed = "completed"   # Finished successfully
    cancelled = "cancelled"   # Aborted or not executed

# ====================================================
# Timesheet Entry Mapping Model
# ====================================================
class TimesheetEntryBase(BaseModel):
    workforce_id: int = Field(..., description="Unique identifier for the workforce member")
    service_id: int = Field(..., description="Unique identifier for the service being provided")
    vehicle_id: Optional[int] = Field(None, description="Unique identifier for the assigned vehicle (if applicable)")
    
    # Shift or entry timings; for on-demand entries these may be left null
    shift_start: Optional[datetime] = Field(None, description="Start time for the work entry or shift")
    shift_end: Optional[datetime] = Field(None, description="End time for the work entry or shift")
    
    # Type of entry and its current status
    timesheet_type: TimesheetType = Field(..., description="Indicates whether this is a scheduled or on-demand entry")
    status: TimesheetStatus = Field(default=TimesheetStatus.pending, description="Current status of the timesheet entry")
    
    # Additional information or special instructions
    notes: Optional[str] = Field("", description="Additional notes or instructions for the timesheet entry")
    
    # Record metadata
    created_date: datetime = Field(default_factory=datetime.utcnow, description="Timestamp when the entry was created")

    class Config:
        orm_mode = True

# Model for creating a new timesheet entry (all fields required as per business rules)
class TimesheetEntryCreate(TimesheetEntryBase):
    pass

# Model for updating an existing timesheet entry; all fields are optional to allow partial updates
class TimesheetEntryUpdate(BaseModel):
    workforce_id: Optional[int] = Field(None, description="Updated workforce identifier")
    service_id: Optional[int] = Field(None, description="Updated service identifier")
    vehicle_id: Optional[int] = Field(None, description="Updated vehicle identifier")
    shift_start: Optional[datetime] = Field(None, description="Updated start time")
    shift_end: Optional[datetime] = Field(None, description="Updated end time")
    timesheet_type: Optional[TimesheetType] = Field(None, description="Updated timesheet type")
    status: Optional[TimesheetStatus] = Field(None, description="Updated timesheet status")
    notes: Optional[str] = Field(None, description="Updated notes or instructions")

# Model for reading a timesheet entry, with a unique timesheet entry identifier
class TimesheetEntryRead(TimesheetEntryBase):
    timesheet_entry_id: int = Field(..., description="Unique identifier for the timesheet entry")
